import math
import matplotlib.pyplot as plt

def convert_gal_mile_to_l_km(gal_per_mile):
    liters_per_km = gal_per_mile * (3.78541 / 1.60934)
    return liters_per_km

def convert_lb_to_kg(lb):
    kg = lb * 0.453592
    return kg

def calculate_fuel_consumption(mlw_lb, distance_per_flight, num_flights_per_year, lifetime):
    landing_gear_percentage = 0.04
    wing_percentage = 0.27
    fuselage_percentage = 0.19
    v_stabilizer_percentage = 0.06
    h_stabilizer_percentage = 0.11
    engine_percentage = 0.20
    accessories_percentage = 0.13

    landing_gear_weight = landing_gear_percentage * mlw_lb
    wing_weight = wing_percentage * mlw_lb
    fuselage_weight = fuselage_percentage * mlw_lb
    v_stabilizer_weight = v_stabilizer_percentage * mlw_lb
    h_stabilizer_weight = h_stabilizer_percentage * mlw_lb
    engine_weight = engine_percentage * mlw_lb
    accessories_weight = accessories_percentage * mlw_lb

    F_gal_per_mile = 1.62 + 0.0098 * (mlw_lb / 1000)
    F_liters_per_km = convert_gal_mile_to_l_km(F_gal_per_mile)

    print(f"F_gal_per_mile: {F_gal_per_mile}")
    print(f"F_liters_per_km: {F_liters_per_km}")

    Fuseaircraft = distance_per_flight * num_flights_per_year * lifetime * F_liters_per_km

    print(f"Distance per flight: {distance_per_flight}")
    print(f"Number of flights per year: {num_flights_per_year}")
    print(f"Lifetime: {lifetime}")
    print(f"Fuseaircraft: {Fuseaircraft}")

    landing_gear_fuel_consumption = Fuseaircraft * (landing_gear_weight / mlw_lb)
    wing_fuel_consumption = Fuseaircraft * (wing_weight / mlw_lb)
    fuselage_fuel_consumption = Fuseaircraft * (fuselage_weight / mlw_lb)
    v_stabilizer_fuel_consumption = Fuseaircraft * (v_stabilizer_weight / mlw_lb)
    h_stabilizer_fuel_consumption = Fuseaircraft * (h_stabilizer_weight / mlw_lb)
    engine_fuel_consumption = Fuseaircraft * (engine_weight / mlw_lb)
    accessories_fuel_consumption = Fuseaircraft * (accessories_weight / mlw_lb)

    return {
        "Fuseaircraft": Fuseaircraft,
        "Landing Gear": landing_gear_fuel_consumption,
        "Wing": wing_fuel_consumption,
        "Fuselage": fuselage_fuel_consumption,
        "Vertical Stabilizer": v_stabilizer_fuel_consumption,
        "Horizontal Stabilizer": h_stabilizer_fuel_consumption,
        "Engine": engine_fuel_consumption,
        "Accessories": accessories_fuel_consumption
    }

def calculate_eol_costs(mlw_lb):
    mlw_kg = convert_lb_to_kg(mlw_lb)

    landing_gear_percentage = 0.04
    wing_percentage = 0.27
    fuselage_percentage = 0.19
    v_stabilizer_percentage = 0.06
    h_stabilizer_percentage = 0.11
    engine_percentage = 0.20
    accessories_percentage = 0.13

    landing_gear_weight = landing_gear_percentage * mlw_kg
    wing_weight = wing_percentage * mlw_kg
    fuselage_weight = fuselage_percentage * mlw_kg
    v_stabilizer_weight = v_stabilizer_percentage * mlw_kg
    h_stabilizer_weight = h_stabilizer_percentage * mlw_kg
    engine_weight = engine_percentage * mlw_kg
    accessories_weight = accessories_percentage * mlw_kg

    eol_costs = {
        "Landing Gear": {
            "weight": landing_gear_weight,
            "wrecycling": 0.50 * landing_gear_weight,
            "wlandfill": 0.30 * landing_gear_weight,
            "wincineration": 0.20 * landing_gear_weight,
            "cost_recycling": -0.75,
            "cost_landfill": 0.2,
            "cost_incineration": 1
        },
        "Wing": {
            "weight": wing_weight,
            "wrecycling": 0.65 * wing_weight,
            "wlandfill": 0.35 * wing_weight,
            "wincineration": 0.05 * wing_weight,
            "cost_recycling": -1.50,
            "cost_landfill": 8,
            "cost_incineration": 0.2
        },
        "Fuselage": {
            "weight": fuselage_weight,
            "wrecycling": 0.65 * fuselage_weight,
            "wlandfill": 0.30 * fuselage_weight,
            "wincineration": 0.05 * fuselage_weight,
            "cost_recycling": -2.25,
            "cost_landfill": 8,
            "cost_incineration": 0.2
        },
        "Horizontal Stabilizer": {
            "weight": h_stabilizer_weight,
            "wrecycling": 0.65 * h_stabilizer_weight,
            "wlandfill": 0.30 * h_stabilizer_weight,
            "wincineration": 0.05 * h_stabilizer_weight,
            "cost_recycling": -1.50,
            "cost_landfill": 8,
            "cost_incineration": 0.2
        },
        "Vertical Stabilizer": {
            "weight": v_stabilizer_weight,
            "wrecycling": 0.65 * v_stabilizer_weight,
            "wlandfill": 0.30 * v_stabilizer_weight,
            "wincineration": 0.05 * v_stabilizer_weight,
            "cost_recycling": -1.50,
            "cost_landfill": 8,
            "cost_incineration": 0.2
        },
        "Engine": {
            "weight": engine_weight,
            "wrecycling": 0.60 * engine_weight,
            "wlandfill": 0.30 * engine_weight,
            "wincineration": 0.10 * engine_weight,
            "cost_recycling": -2.25,
            "cost_landfill": 8,
            "cost_incineration": 0.2
        },
        "Accessories": {
            "weight": accessories_weight,
            "wrecycling": 0.65 * accessories_weight,
            "wlandfill": 0.30 * accessories_weight,
            "wincineration": 0.05 * accessories_weight,
            "cost_recycling": -1.50,
            "cost_landfill": 8,
            "cost_incineration": 0.2
        }
    }

    total_eol_costs = {}
    for component, values in eol_costs.items():
        total_cost = (values["wrecycling"] * values["cost_recycling"] +
                      values["wlandfill"] * values["cost_landfill"] +
                      values["wincineration"] * values["cost_incineration"])
        total_eol_costs[component] = total_cost

    return total_eol_costs

def main():
    # Input values
    mlw_lb = float(input("Enter the Maximum Landing Weight (MLW) in lb: "))
    distance_per_flight = float(input("Enter the distance per flight in kilometers: "))
    num_flights_per_year = int(input("Enter the number of flights per year: "))
    lifetime = 25  # years
    fuel_price_per_liter = 0.224  # euros per liter of Kerosene

    # Calculate fuel consumption for each component
    fuel_consumption = calculate_fuel_consumption(mlw_lb, distance_per_flight, num_flights_per_year, lifetime)

    # Calculate use phase costs for each component
    use_phase_costs = {component: consumption * fuel_price_per_liter for component, consumption in fuel_consumption.items() if component != "Fuseaircraft"}

    # Calculate EoL costs for each component
    eol_costs = calculate_eol_costs(mlw_lb)

    # Print results
    print(f"\nTotal fuel consumption for the aircraft over its lifetime (Fuseaircraft): {fuel_consumption['Fuseaircraft']:.2f} liters")
    print("\nFuel consumption for each component over the aircraft's lifetime:")
    for component, consumption in fuel_consumption.items():
        if component != "Fuseaircraft":
            print(f"{component}: {consumption:.2f} liters")

    print("\nUse phase costs for each component:")
    for component, cost in use_phase_costs.items():
        print(f"{component}: €{cost:.2f}")

    print("\nEnd-of-Life (EoL) costs for each component:")
    for component, cost in eol_costs.items():
        print(f"{component}: €{cost:.2f}")

    # Calculate total aircraft cost
    total_use_cost = sum(use_phase_costs.values())
    total_eol_cost = sum(eol_costs.values())
    total_aircraft_cost = total_use_cost + total_eol_cost

    print(f"\nTotal Aircraft Cost: €{total_aircraft_cost:.2f}")

  # Plotting the Use Phase Costs using a line chart
    components = list(use_phase_costs.keys())
    use_costs = list(use_phase_costs.values())

    fig, ax = plt.subplots()
    ax.plot(components, use_costs, marker='o', linestyle='-', color='b', label='Use Phase Costs')

    ax.set_xlabel('Components')
    ax.set_ylabel('Costs (€)')
    ax.set_title('Use Phase Costs for Aircraft Components')
    ax.set_xticklabels(components, rotation=45, ha='right')

    plt.tight_layout()
    plt.legend()
    plt.show()

    # Plotting the EoL Costs using a line chart
    eol_costs_values = list(eol_costs.values())
    fig, ax = plt.subplots()
    ax.plot(components, eol_costs_values, marker='o', linestyle='-', color='r', label='EoL Costs')

    ax.set_xlabel('Components')
    ax.set_ylabel('Costs (€)')
    ax.set_title('End-of-Life (EoL) Costs for Aircraft Components')
    ax.set_xticklabels(components, rotation=45, ha='right')

    plt.tight_layout()
    plt.legend()
    plt.show()

if __name__ == "__main__":
    main()